//--------------------------------------------------------------------
// FILENAME:			scandisp.c
//
// Copyright(c) 1999-2002 Symbol Technologies Inc. All rights reserved.
//
// DESCRIPTION:			Contains window display functions for the main 
//						application windows.  This is demo code. Use at your own risk.
//
// NOTES:	Some Routines in this file were taken from a terminal imager demo app.				
//
//--------------------------------------------------------------------


#include <windows.h>

#include "scandisp.h"

#define BUFFERSIZE 1024 + sizeof(BITMAPINFOHEADER) + 1


#define MAX_TIFF_FIELDS 14  // the current rev of goldeneye sw uses this
#define TIFF_NEW_SUBFILE_TAG	0x00fe
#define TIFF_IMG_WIDTH_TAG		0x0100
#define TIFF_IMG_LEN_TAG		0x0101
#define TIFF_BPP_TAG				0x0102
#define TIFF_COMPRESS_TAG		0x0103
#define TIFF_PHOTOMETRIC_TAG	0x0106
#define TIFF_STRIP_OFFSET_TAG	0x0111
#define TIFF_ROWSPERSTRIP_TAG	0x0116
#define TIFF_STRIPBYTECNT_TAG	0x0117
#define TIFF_XRES_TAG			0x011a
#define TIFF_YRES_TAG			0x011b
#define TIFF_RES_UNIT_TAG		0x0128
#define TIFF_SOFTWARE_TAG		0x0131
#define TIFF_TERMINATION_TAG	0x0000


enum {
	 TIFF_NEW_SUBFILE_INDX,
 TIFF_IMG_WIDTH_INDX,
 TIFF_IMG_LEN_INDX,
 TIFF_BPP_INDX,
 TIFF_COMPRESS_INDX,
 TIFF_PHOTOMETRIC_INDX,
 TIFF_STRIP_OFFSET_INDX,
 TIFF_ROWSPERSTRIP_INDX,
 TIFF_STRIPBYTECNT_INDX,
 TIFF_XRES_INDX,
 TIFF_YRES_INDX,
 TIFF_RES_UNIT_INDX,
 TIFF_SOFTWARE_INDX
};




HANDLE      g_hImage                = NULL;
HPALETTE    g_hPalette              = NULL;


static unsigned char buff[BUFFERSIZE];



static UINT BitsPerPixel (HANDLE hImage);
static HPALETTE CreateExactPalette (HANDLE hImage);



// Input a handle to the device context, handle to the dib, picturesize multiplier, upper left corner for image display,
// ... and the width and height of the image display.  
//
// Stretches image onto device.
//

void DisplayDIBSection (HDC hdc, HANDLE hImage, int picturesize, int topx, int topy, int windowcx, int windowcy)
{
   HDC hdcMem;
   HBITMAP hOldBitmap;
   DIBSECTION ds;
	int newheight;
	int newwidth;
   int cx, cy;
	HPALETTE    hPalette = g_hPalette;
	double ratio;

   hdcMem = CreateCompatibleDC (hdc);
   if (hPalette != NULL) 
	{
       SelectPalette (hdcMem, hPalette, FALSE);
       RealizePalette (hdcMem);
       SelectPalette (hdc, hPalette, FALSE);
       RealizePalette (hdc);
   }

   GetObject (hImage, sizeof (ds), &ds);
   cx = ds.dsBmih.biWidth;
   cy = ds.dsBmih.biHeight;


	if(picturesize == 0)  // don't stretch it
	{
		newheight = cy;
		newwidth = cx;
	}
	else if (picturesize == 1) // try making it twice as high and wide
	{
		newheight = cy * 2;
		newwidth = cx * 2;
	}
	else
	{
		newheight = cy * 3;	// try making it three times as high and wide
		newwidth = cx * 3;
	}

	
	if(newwidth > windowcx)		// If desired width or height is larger than available display width, make it fit but keep 
	{									// ...the aspect ratio the same as the original image.
		ratio = (double)windowcx/newwidth;
		
		newwidth = windowcx;
		newheight = (int)(ratio * newheight);
	}
	if(newheight > windowcy)
	{
		ratio = (double)windowcy/newheight;
		
		newheight = windowcy;
		newwidth = (int)(ratio * newwidth);
	}

	hOldBitmap = (HBITMAP)SelectObject (hdcMem, hImage);

	SetStretchBltMode(hdc, COLORONCOLOR); 

	StretchBlt (hdc,  topx, topy, newwidth, newheight, hdcMem, 0,0,cx,cy, SRCCOPY); // now stretch it onto the device


   SelectObject (hdcMem, hOldBitmap);
   DeleteDC (hdcMem);
}



// from the jpeg image sent from the scanner, create a dib section
#define WIDTHBYTES(bits)    (((bits) + 31) / 32 * 4)
HANDLE CreateDIBSectionFromRawBits( int RawImageColor, unsigned RawImageWidth, unsigned RawImageHeight, 
											  unsigned RawImageBitsPerPixel, BYTE *RawImageRaw, HWND hWnd )
{
   HANDLE hImage;
	BITMAPINFOHEADER bmiHeader;
	BITMAPINFOHEADER FAR *lpbi;
	LPSTR pDIB;
	LPSTR pHDR;
	LPSTR pRAW;
	DWORD dwUnpaddedRow;
	DWORD dwPaddedRow;
	DWORD dwPad;
	DWORD dwColorMapSize;
	int i;
	WORD  j, k;
	LPVOID   lpvBits;
	HDC hdc;
	int n;


	if (RawImageColor)
	{
		/* for 24-Bit color, each pixel is 3 bytes */
		dwUnpaddedRow = RawImageWidth * 3;
		dwColorMapSize = 0; //no color table
	}
	else
	{
		dwUnpaddedRow = RawImageWidth;
		dwColorMapSize = 1024;
	}

	/* Each row in a DIB is padded to 4 bytes boundary */ 	
	dwPaddedRow = (( dwUnpaddedRow + 0x3) & ~0x3);
	dwPad = dwPaddedRow - dwUnpaddedRow;

	bmiHeader.biSize               = sizeof(BITMAPINFOHEADER);
   bmiHeader.biWidth              = RawImageWidth;
   bmiHeader.biHeight             = RawImageHeight;
   bmiHeader.biPlanes             = 1;
   bmiHeader.biBitCount           = (WORD)RawImageBitsPerPixel;
   bmiHeader.biCompression        = 0;
   bmiHeader.biSizeImage          = 0;
   bmiHeader.biXPelsPerMeter      = 0;
   bmiHeader.biYPelsPerMeter      = 0;
   bmiHeader.biClrUsed            = 0;
   bmiHeader.biClrImportant       = 0;

	lpbi = (BITMAPINFOHEADER FAR *)buff;// do instead of global allocation
	*lpbi = bmiHeader;

	pHDR = (LPSTR)lpbi;
	k = (WORD)bmiHeader.biSize ;

	/* build the RGBQUAD with a gray level color table*/
	if (dwColorMapSize > 0)
	{
		for (i = 0; i < 256; i++)
		{
			pHDR[k++] = (char) i;		//Blue
			pHDR[k++] = (char) i;		//Green
			pHDR[k++] = (char) i;		//Red
			pHDR[k++] = 0;		//Reserved
		}
	}

	lpvBits  = NULL;

	hdc = GetDC(hWnd);

	/* for debugging */	
	n = GetDeviceCaps(hdc,NUMCOLORS);
	n = GetDeviceCaps(hdc,BITSPIXEL);
	
	/* end for debugging */

   hImage = CreateDIBSection(hdc,
                               (LPBITMAPINFO)pHDR,
                               DIB_RGB_COLORS,
                               &lpvBits,
                               0,
                               0L);

	if( hImage != NULL && lpvBits != NULL)
	{

		//Copy the DIB bits from the user buffer into the DIB

		pDIB = (LPSTR)lpvBits;
		pRAW = (LPSTR)RawImageRaw;

		for ( j=0; j<RawImageHeight; j++)
		{
			int nDepthInOffset = j*RawImageWidth*3;
			int nDepthOutOffset = (RawImageHeight-j-1)*WIDTHBYTES(RawImageWidth*24); 
			for ( i=0; i< (int) RawImageWidth; i++)
			{
				int nInOffset = nDepthInOffset + i*3;
				int nOutOffset = nDepthOutOffset + i*3;  
				pDIB[nOutOffset]   = pRAW[nInOffset];
				pDIB[nOutOffset+1] = pRAW[nInOffset+1];
				pDIB[nOutOffset+2] = pRAW[nInOffset+2];
			}
		
		}
	}
	return hImage;
}


// from DIB file sent from imager, , create a dib section
HANDLE CreateDIBSectionFromDIBFile( unsigned long length, 
											   BYTE *RawImageRaw, HWND hWnd )
{
   HANDLE hImage = NULL;
	BITMAPFILEHEADER bmiFileHeader;
	BITMAPFILEHEADER FAR *lpfbi;
	BITMAPINFOHEADER bmiHeader;
	BITMAPINFOHEADER FAR *lpbi;
	LPSTR pHDR;
	LPSTR pRAW;
	LPVOID   lpvBits;
	HDC hdc;
	int n;

	lpfbi = (BITMAPFILEHEADER FAR *)RawImageRaw; // point to bmp file header in input stream
	bmiFileHeader = *lpfbi++;							// store file heaader
	lpbi = (BITMAPINFOHEADER FAR *)lpfbi;			// point to bmp info header in input stream
	pRAW = (BYTE *)RawImageRaw + bmiFileHeader.bfOffBits;	 // point to bitmap bits in input data stream

	bmiHeader = *lpbi++;									// store bmp info header

	lpbi = (BITMAPINFOHEADER FAR *)buff;			// do instead of global allocation - point to beginning of storage buffer


	if((bmiFileHeader.bfOffBits - sizeof(BITMAPFILEHEADER)) <= BUFFERSIZE)
	{
		// copies color table too if there is one
		memcpy((void *)lpbi, (void *)(RawImageRaw + sizeof(BITMAPFILEHEADER)), (bmiFileHeader.bfOffBits - sizeof(BITMAPFILEHEADER))); 
		pHDR = (LPSTR)lpbi;	
		lpvBits  = NULL;

		hdc = GetDC(hWnd);

		/* for debugging */	
		n = GetDeviceCaps(hdc,NUMCOLORS);
		n = GetDeviceCaps(hdc,BITSPIXEL);
	
		/* end for debugging */

		hImage = CreateDIBSection(hdc,
                               (LPBITMAPINFO)pHDR,
                               DIB_RGB_COLORS,
                               &lpvBits,
                               0,
                               0L);
		//Copy the DIB bits from the user buffer into the DIB

		if( hImage != NULL && lpvBits != NULL)
			memcpy(lpvBits, pRAW, bmiFileHeader.bfSize - bmiFileHeader.bfOffBits);

	}
	return hImage;
}




// from TIFF file from imager, create a dib section
HANDLE CreateDIBSectionFromTIFFFile( unsigned long l, 
											   BYTE *w, HWND hWnd )
{
	struct tiftags{
		WORD tag;
		WORD data_type;
		DWORD length;
		DWORD data_or_ptr;
	};
	int i,j,k,n;
	BYTE *pRaw = w;
   HANDLE hImage = NULL;
	BITMAPINFOHEADER bmiHeader;
	BITMAPINFOHEADER FAR *lpbi;
	LPSTR pDib;
	LPSTR pHDR;
	DWORD dwUnpaddedRow;
	DWORD dwPaddedRow;
	DWORD dwPad;

	DWORD dwColorMapSize;
	LPVOID   lpvBits;
	HDC hdc;
	DWORD dwDataSize;
	DWORD RawImageWidth;
	DWORD RawImageHeight;
	DWORD RawImageBitsPerPixel;
	
	
	
	struct field_data_struct{
		WORD field_tag;
		int value_indx;
	}valid_tags[MAX_TIFF_FIELDS] = {
		{TIFF_NEW_SUBFILE_TAG, -1},
		{TIFF_IMG_WIDTH_TAG, -1},
		{TIFF_IMG_LEN_TAG, -1},
		{TIFF_BPP_TAG, -1},
		{TIFF_COMPRESS_TAG, -1},
		{TIFF_PHOTOMETRIC_TAG, - 1},
		{TIFF_STRIP_OFFSET_TAG, -1},
		{TIFF_ROWSPERSTRIP_TAG, - 1},
		{TIFF_STRIPBYTECNT_TAG, -1},
		{TIFF_XRES_TAG, -1},
		{TIFF_YRES_TAG, -1},
		{TIFF_RES_UNIT_TAG, -1},
		{TIFF_SOFTWARE_TAG, -1},
		{TIFF_TERMINATION_TAG, -1}
 };

	int field_index[MAX_TIFF_FIELDS] = {0};


	struct tiftags mytif[MAX_TIFF_FIELDS];

	BYTE *pTifTags = (BYTE *)(w + 8);

	unsigned ntiftags = (*pTifTags) << 8;
	pTifTags++;
	ntiftags |= *pTifTags++;
	if(ntiftags != MAX_TIFF_FIELDS)
		return NULL;



	for(i = 0; i < (int)ntiftags; i++)
	{
		mytif[i].tag = (*pTifTags) << 8;
		pTifTags++;
		mytif[i].tag |= *pTifTags++;
		
		mytif[i].data_type = (*pTifTags) << 8;
		pTifTags++;
		mytif[i].data_type |= *pTifTags++;

		mytif[i].length = (*pTifTags) << 24;
		pTifTags++;
		mytif[i].length |= (*pTifTags) << 16;
		pTifTags++;
		mytif[i].length |= (*pTifTags) << 8;
		pTifTags++;
		mytif[i].length |= *pTifTags++;

		mytif[i].data_or_ptr = (*pTifTags) << 24;
		pTifTags++;
		mytif[i].data_or_ptr |= (*pTifTags) << 16;
		pTifTags++;
		mytif[i].data_or_ptr |= (*pTifTags) << 8;
		pTifTags++;
		mytif[i].data_or_ptr |= *pTifTags++;



		for(j = 0; j <  MAX_TIFF_FIELDS - 1; j++)
			if(mytif[i].tag == valid_tags[j].field_tag)
			{
				valid_tags[j].value_indx = i;
				break;
			}

		if((j == (MAX_TIFF_FIELDS-1)) && (i == ((int)ntiftags - 1)))   // termination tag
		{
			if(mytif[i].tag != 0) 
				return NULL;
		}
	}
	for(j = 0; j <  MAX_TIFF_FIELDS - 1; j++)
		if(valid_tags[j].value_indx == -1)
			return NULL; // didn't get all the fields we expected - one must have been a  duplicate


	// now we can build the bitmap for display

	
	pRaw += mytif[valid_tags[TIFF_STRIP_OFFSET_INDX].value_indx].data_or_ptr;

	dwDataSize = mytif[valid_tags[TIFF_STRIPBYTECNT_INDX].value_indx].data_or_ptr;

	
	RawImageWidth = mytif[valid_tags[TIFF_IMG_WIDTH_INDX].value_indx].data_or_ptr;
	RawImageHeight =mytif[valid_tags[TIFF_IMG_LEN_INDX].value_indx].data_or_ptr;
	RawImageBitsPerPixel = 	(WORD)(mytif[valid_tags[TIFF_BPP_INDX].value_indx].data_or_ptr >> 16);


	

	dwColorMapSize = 1024;
	dwUnpaddedRow = RawImageWidth;
	

	/* Each row in a DIB is padded to 4 bytes boundary */ 	
	dwPaddedRow = (( dwUnpaddedRow + 0x3) & ~0x3);
	dwPad = dwPaddedRow - dwUnpaddedRow;



   bmiHeader.biSize               = sizeof(BITMAPINFOHEADER);
   bmiHeader.biWidth              = RawImageWidth;  // in pixels
   bmiHeader.biHeight             = RawImageHeight; // in pixels
   bmiHeader.biPlanes             = 1;
   bmiHeader.biBitCount           = (WORD)RawImageBitsPerPixel;
   bmiHeader.biCompression        = 0;
   bmiHeader.biSizeImage          = 0;
   bmiHeader.biXPelsPerMeter      = 0;
   bmiHeader.biYPelsPerMeter      = 0;
   bmiHeader.biClrUsed            = 0;
   bmiHeader.biClrImportant       = 0;



	lpbi = (BITMAPINFOHEADER FAR *)buff;			// do instead of global allocation - point to beginning of storage buffer
	pHDR = (LPSTR)lpbi;	
	*lpbi++ = bmiHeader;

	k = (WORD)bmiHeader.biSize ;

	/* build the RGBQUAD with a gray level color table*/
	if (dwColorMapSize > 0){
		for (i = 0; i < 256; i++){
			pHDR[k++] = (char) i;		//Blue
			pHDR[k++] = (char) i;		//Green
			pHDR[k++] = (char) i;		//Red
			pHDR[k++] = 0;		//Reserved
		}
	}

	lpvBits  = NULL;
	hdc = GetDC(hWnd);

		/* for debugging */	
	n = GetDeviceCaps(hdc,NUMCOLORS);
	n = GetDeviceCaps(hdc,BITSPIXEL);
	
		/* end for debugging */

	hImage = CreateDIBSection(hdc,
                               (LPBITMAPINFO)pHDR,
                               DIB_RGB_COLORS,
                               &lpvBits,
                               0,
                               0L);
	//Copy the DIB bits from the user buffer into the DIB

		
	if( hImage != NULL && lpvBits != NULL)
	{

		int RawImageColor;
		DWORD dwBtmOfRaw;
		k = 0;
		pDib = (LPSTR)lpvBits;
		RawImageColor = 8;
		
		for ( i = (RawImageHeight - 1); i >= 0; i--)
		{
			dwBtmOfRaw = (i * dwUnpaddedRow  ) ;
			if(RawImageColor == 1)
			{ 
				for (l = 0; l < dwUnpaddedRow; l+=3)
				{
					pDib[k++] = pRaw[dwBtmOfRaw + l+2];
					pDib[k++] = pRaw[dwBtmOfRaw + l+1];
					pDib[k++] = pRaw[dwBtmOfRaw + l];
				
				}
			}
			else
			{
				for (l = 0; l < dwUnpaddedRow; l++)
					pDib[k++] = pRaw[dwBtmOfRaw + l];
				
			}
			for ( j = 0; j < (int)dwPad; j++)
				pDib[k++] = 0;
		}

	}

	return hImage;


}


HPALETTE GetPaletteHandle (HWND hwnd, HANDLE hImage)
{
    HDC hdc;
    HPALETTE hPalette;
    int bpp;

    // Return NULL if this isn't a palette-based device
	 if(!(hwnd && hImage))
		 return NULL;

    hdc = GetDC (hwnd);
    if (!(GetDeviceCaps (hdc, RASTERCAPS) & RC_PALETTE)) 
	 {
        ReleaseDC (hwnd, hdc);
        return NULL;
    }

    // Otherwise create a palette and return the palette handle
    bpp = BitsPerPixel (hImage);
    if (bpp <= 8)
        hPalette = CreateExactPalette (hImage);
    else if ((bpp == 16) || (bpp == 24) || (bpp == 32)) 
	 {
		  hPalette = CreateHalftonePalette(hdc);
    }

    ReleaseDC (hwnd, hdc);
    return hPalette;
}

UINT BitsPerPixel (HANDLE hImage)
{
    DIBSECTION ds;

    GetObject (hImage, sizeof (ds), &ds);
    return (UINT) ds.dsBmih.biBitCount;
}

static HPALETTE CreateExactPalette (HANDLE hImage)
{
    DIBSECTION ds;
    UINT nColors, i;
    RGBQUAD* prgb;
    LOGPALETTE* plp;
    HDC hdc, hdcMem;
    HBITMAP hOldBitmap;
    HPALETTE hPalette;
    DWORD dwSize;

    // Get the number of colors in the image
    GetObject (hImage, sizeof (ds), &ds);
    if (ds.dsBmih.biClrUsed != 0)
        nColors = ds.dsBmih.biClrUsed;
    else
        nColors = 1 << ds.dsBmih.biBitCount;

    if (nColors > 256) // Sanity check
        return NULL;

    // Retrieve the image's color table
    if ((prgb = (RGBQUAD*) HeapAlloc (GetProcessHeap (), 0,
        nColors * sizeof (RGBQUAD))) == NULL)
        return NULL;

    hdc = GetDC (NULL);
    hdcMem = CreateCompatibleDC (hdc);
    hOldBitmap = (HBITMAP )SelectObject (hdcMem, hImage);
    nColors = min (nColors, GetDIBColorTable (hdcMem, 0, nColors, prgb));
	//GetDIBColorTable is not supported in CE


    SelectObject (hdcMem, hOldBitmap);
    DeleteDC (hdcMem);
    ReleaseDC (NULL, hdc);

    if (nColors == 0) 
	 { // Another sanity check
        HeapFree (GetProcessHeap (), 0, prgb);
        return NULL;
    }
        
    // Create a logical palette from the colors in the color table
    dwSize = sizeof (LOGPALETTE) + ((nColors - 1) * sizeof (PALETTEENTRY));
    if ((plp = (LOGPALETTE*) HeapAlloc (GetProcessHeap (), 0, dwSize)) == NULL) 
	 {
        HeapFree (GetProcessHeap (), 0, prgb);
        return NULL;
    }

    plp->palVersion = 0x300;
    plp->palNumEntries = (WORD) nColors;

    for (i=0; i<nColors; i++) 
	 {
        plp->palPalEntry[i].peRed   = prgb[i].rgbRed;
        plp->palPalEntry[i].peGreen = prgb[i].rgbGreen;
        plp->palPalEntry[i].peBlue  = prgb[i].rgbBlue;
        plp->palPalEntry[i].peFlags = 0;
    }

    hPalette = CreatePalette (plp);

    HeapFree (GetProcessHeap (), 0, plp);
    HeapFree (GetProcessHeap (), 0, prgb);
    return hPalette;
}

